#ifndef __CMidi__
#define __CMidi__

//	===========================================================================

#include <MathTools/CMathTools.hpp>
#include <Basics/CString.hpp>

//	===========================================================================

using Exponent::MathTools::CMathTools;
using Exponent::Basics::CString;

//	===========================================================================

namespace Exponent
{
	namespace Midi
	{
		/**
		 * @class CMidi CMidi.hpp
		 * @brief Provides midi functions and values
		 *
		 * @date 23/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CMidi.hpp,v 1.7 2007/02/08 21:08:09 paul Exp $
		 */
		class CMidi
		{
		public:

//	===========================================================================

			/**
			 * Convert Midi to a float value
			 * @param midiValue The value midi scaled (0 : 127)
			 * @retval double The normalised double value (0 : 1)
			 */
			FORCEINLINE static double midiToFloat(const long midiValue)
			{
				const static double midiScalar = 0.00787401574803149606299212598425197;
				return CMathTools::clamp(midiValue * midiScalar);
			}

			/**
			 * Convert a normalised Float to midi value
			 * @param value The value to convert (0 : 1)
			 * @retval long The midi value (0 : 127)
			 */
			FORCEINLINE static long floatToMidi(const double value)
			{
				const static double floatScalar = 127.0;
				return (long)(value * floatScalar);
			}

			/**
			 * Get a midi note stirng
			 * @param midiNote The midi note to convert to a string
			 * @param theString The stirng to write to
			 */
			static void getMidiNoteString(const long midiNote, CString &theString)
			{
				if (midiNote >= 0 && midiNote < 128)
				{
					const static char *CMIDI_NOTE_STRINGS[] = { "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "Bb", "B" };
					theString.setStringWithFormat("%s%li", CMIDI_NOTE_STRINGS[midiNote % 12], (midiNote / 12) - 1);
				}
				else
				{
					theString = "Range Error";
				}
			}

			/**
			 * Is the note a white note
			 * @param midiNote The note to check
			 * @retval bool True if the note is a white note, flse if black note
			 */
			static bool isWhiteNote(const long midiNote)
			{
				const long index = midiNote % 12;
				return (!(index == 1 || index == 3 || index == 5 || index == 8 || index == 10));
			}

			/**
			 * Is the note a black note
			 * @param midiNote The note to check
			 * @retval bool True if the note is a black note, flse if white note
			 */
			static bool isBlackNote(const long midiNote)
			{
				const long index = midiNote % 12;
				return (index == 1 || index == 3 || index == 5 || index == 8 || index == 10);
			}

//	===========================================================================

			const static long CMIDI_MIDI_NONE					= 0x0;		/**< No midi information */

//	===========================================================================

			const static long CMIDI_NOTE_OFF 					= 0x80;		/**< Note off */
			const static long CMIDI_NOTE_ON	 					= 0x90;		/**< Note on */
			const static long CMIDI_POLY_PRESSURE	 			= 0xA0;		/**< AKA Polyphonic Aftertouch */
			const static long CMIDI_CONTROL_CHANGE 				= 0xB0;		/**< Control change */
			const static long CMIDI_PROGRAM_CHANGE 				= 0xC0;		/**< Program change */
			const static long CMIDI_CHANNEL_PRESSURE 			= 0xD0;		/**< AKA Aftertouch */
			const static long CMIDI_PITCH_BEND 					= 0xE0;		/**< Pitch bend */

//	===========================================================================

			// Sysex and more
			const static long CMIDI_SYSEX						= 0xF0;		/**< Sysex */
			const static long CMIDI_SYSTEM_COMMON 				= 0xF1;		/**< Undefined System Common */
			const static long CMIDI_SONG_POSITION 				= 0xF2;		/**< Song position */
			const static long CMIDI_SONG_SELECT	 				= 0xF3;		/**< Song select */
			const static long CMIDI_F4	 						= 0xF4;		/**< UNDEFINED IN m_midi SPEC */
			const static long CMIDI_F5	 						= 0xF5;		/**< UNDEFINED IN m_midi SPEC */
			const static long CMIDI_TUNER_QUEST	 				= 0xF6;		/**< Tuner request */
			const static long CMIDI_EOX 						= 0xF7;		/**< End OF SysEx */
			const static long CMIDI_TIMING_CLOCK 				= 0xF8;		/**< Timing clock */
			const static long CMIDI_F9	 						= 0xF9;		/**< UNDEFINED IN m_midi SPEC */
			const static long CMIDI_START 						= 0xFA;		/**< Midi start */
			const static long CMIDI_CONTINUE 					= 0xFB;		/**< Midi continue */
			const static long CMIDI_STOP						= 0xFC;		/**< Midi stop */
			const static long CMIDI_FD	 						= 0xFD;		/**< UNDEFINED IN m_midi SPEC */
			const static long CMIDI_ACTIVE_SENSING				= 0xFE;		/**< Active sensing */
			const static long CMIDI_SYSTEM_RESET 				= 0xFF;		/**< System reset */

//	===========================================================================

			// control numbers for m_midi_CONTROL_CHANGE (m_midi status byte 0xB0)

			// MSBs
			const static long CMIDI_BACK_CHANGE					= 0x00;		/**< Bank change */
			const static long CMIDI_MODULATION_WHEEL			= 0x01;		/**< Modulation wheel */
			const static long CMIDI_BREATH_CONTROL	 			= 0x02;		/**< Breath control */
			const static long CMIDI_UNDEFINED_CC03	 			= 0x03;		/**< Undefined CC03 */
			const static long CMIDI_FOOT_CONTROLLER				= 0x04;		/**< Foot controller */
			const static long CMIDI_PORTAMENTO_TIME	 			= 0x05;		/**< Portamento time */
			const static long CMIDI_DATA_ENTRY	 				= 0x06;		/**< Data entry */
			const static long CMIDI_CHANNEL_VOLUME	 			= 0x07;		/**< Channel volume */
			const static long CMIDI_BALANCE						= 0x08;		/**< Balance */
			const static long CMIDI_UNDEFINED_CC09	 			= 0x09;		/**< Undefined CC09 */
			const static long CMIDI_PAN 						= 0x0A;		/**< Pan */
			const static long CMIDI_EXPRESSION     				= 0x0B;		/**< Expression */
			const static long CMIDI_EFFECT_CONTROL1	 			= 0x0C;		/**< Effect 1 */
			const static long CMIDI_EFFECT_CONTROL2	 			= 0x0D;		/**< Effect 2 */
			const static long CMIDI_UNDEFINED_CC0E 				= 0x0E;		/**< Undefined CC0E */
			const static long CMIDI_UNDEFINED_CC0F 				= 0x0F;		/**< Undefined CC0F */
			const static long CMIDI_GENERAL_PURPOSE1 			= 0x10;		/**< General purpose 1 */
			const static long CMIDI_GENERAL_PURPOSE2 			= 0x11;		/**< General purpose 2 */
			const static long CMIDI_GENERAL_PURPOSE3 			= 0x12;		/**< General purpose 3 */
			const static long CMIDI_GENERAL_PURPOSE4 			= 0x13;		/**< General purpose 4 */
			const static long CMIDI_UNDEFINED_CC14 				= 0x14;		/**< Undefined CC14 */
			const static long CMIDI_UNDEFINED_CC15 				= 0x15;		/**< Undefined CC15 */
			const static long CMIDI_UNDEFINED_CC16 				= 0x16;		/**< Undefined CC16 */
			const static long CMIDI_UNDEFINED_CC17 				= 0x17;		/**< Undefined CC17 */
			const static long CMIDI_UNDEFINED_CC18 				= 0x18;		/**< Undefined CC18 */
			const static long CMIDI_UNDEFINED_CC19 				= 0x19;		/**< Undefined CC19 */
			const static long CMIDI_UNDEFINED_CC1A 				= 0x1A;		/**< Undefined CC1A */
			const static long CMIDI_UNDEFINED_CC1B 				= 0x1B;		/**< Undefined CC1B */
			const static long CMIDI_UNDEFINED_CC1C 				= 0x1C;		/**< Undefined CC1C */
			const static long CMIDI_UNDEFINED_CC1D 				= 0x1D;		/**< Undefined CC1D */
			const static long CMIDI_UNDEFINED_CC1E 				= 0x1E;		/**< Undefined CC1E */
			const static long CMIDI_UNDEFINED_CC1F 				= 0x1F;		/**< Undefined CC1F */

//	===========================================================================

			//[0x20:0x3F] are LSBs

			const static long CMIDI_SUSTAIN 					= 0x40;		/**< Sustain */
			const static long CMIDI_PORTAMENTO					= 0x41;		/**< Portamento */
			const static long CMIDI_SOSTENUTO					= 0x42;		/**< Sostenuto */
			const static long CMIDI_SOFT 						= 0x43;		/**< Soft */
			const static long CMIDI_HOLD2	 					= 0x45;		/**< Hold 2 */
			const static long CMIDI_GENERAL_PURPOSE5 			= 0x50;		/**< General purpose 5 */
			const static long CMIDI_GENERAL_PURPOSE6 			= 0x51;		/**< General purpose 6 */
			const static long CMIDI_GENERAL_PURPOSE7 			= 0x52;		/**< General purpose 7 */
			const static long CMIDI_GENERAL_PURPOSE8 			= 0x53;		/**< General purpose 8 */

//	===========================================================================

			const static long CMIDI_EXTERNAL_EFFECTS_DEPTH	 	= 0x5B;		/**< Effect depth */
			const static long CMIDI_TREMELO_DEPTH 				= 0x5C;		/**< Temelo depth */
			const static long CMIDI_CHORUS_DEPTH 				= 0x5D;		/**< Chorus depth */
			const static long CMIDI_CELESTE_DEPTH 				= 0x5E;		/**< Celeste depth */
			const static long CMIDI_PHASE_DEPTH	 				= 0x5F;		/**< Phase depth */
			const static long CMIDI_DATA_INCREMENT 				= 0x60;		/**< Data increment */
			const static long CMIDI_DATA_DECREMENT	 			= 0x61;		/**< Data decrement */
			const static long CMIDI_NON_REG_PARAM_NUMBER_MSB 	= 0x62;		/**< Non registered parametr number lsb */
			const static long CMIDI_NON_REG_PARAM_NUMBER_LSB	= 0x63;		/**< Non registered parametr number msb */
			const static long CMIDI_REG_PARAM_NUMBER_MSB		= 0x64;		/**< Registered parametr number lsb */
			const static long CMIDI_REG_PARAM_NUMBER_LSB        = 0x65;		/**< Registered parametr number ,sb */

//	===========================================================================

			const static long CMIDI_ALL_SOUNDS_OFF 				= 0x78;		/**< All sounds off */
			const static long CMIDI_RESET_ALL_CONTROLLERS		= 0x79;		/**< Reset all controller */
			const static long CMIDI_LOCAL_CONTROL				= 0x7A;		/**< Local control */
			const static long CMIDI_ALL_NOTES_OFF				= 0x7B;		/**< All notes off */
			const static long CMIDI_OMNI_MODE_OFF 				= 0x7C;		/**< Omni mode off */
			const static long CMIDI_OMNI_MODE_ON 				= 0x7D;		/**< Omni mode on */
			const static long CMIDI_MONO_MODE_ON				= 0x7E;		/**< Mono mode on */
			const static long CMIDI_POLY_MODE_ON				= 0x7F;		/**< Poly mode on */

//	===========================================================================

			// Standard m_midi info
			const static long CMIDI_MAX_NOTES					= 128;		/**< Maximum number of midi notes */
			const static long CMIDI_MAX_CC	 					= 128;		/**< Maximum number of midi controllers */
			const static long CMIDI_MAX_NUMBER_OF_NOTES			= 127;		/**< Maximum number of midi notes */

//	===========================================================================

			const static long CMIDI_STATUS_MASK					= 0xF0;		/**< Status mask */
			const static long CMIDI_CHANNEL_MASK				= 0x08;		/**< Channel mask */
			const static long CMIDI_VALUE_MASK					= 0x7F;		/**< Value mask */

//	===========================================================================

			// MIDI FILE VARIABLES
			const static long CMIDI_MIDIFILE_TRACK_NUM			= 0x20;		/**< Midi file track num id */
			const static long CMIDI_MIDIFILE_TEMPO				= 0x51;		/**< Midi file tempo marker */
			const static long CMIDI_MIDIFILE_TIME_SIGNATURE		= 0x58;		/**< Midi file time signature */
			const static long CMIDI_MIDIFILE_TRACK_NAME			= 0x03;		/**< Midi file track name */

		};
	}
}
#endif	// End of CMidi.hpp